package com.artfess.dataShare.dataResource.ods.controller;


import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.model.CommonResult;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.StringUtil;
import com.artfess.base.valid.AddGroup;
import com.artfess.dataShare.dataResource.ods.manager.BizOdsFieldManager;
import com.artfess.dataShare.dataResource.ods.model.BizOdsField;
import com.artfess.dataShare.dataResource.ods.vo.BizOdsTableVo;
import com.artfess.dataShare.dataResource.ods.vo.CatalogTableVo;
import com.artfess.dataShare.dataResource.ods.vo.OdsTableDetailVo;
import com.artfess.dataShare.factory.QueryParamVo;
import com.artfess.dataShare.factory.QueryResultData;
import com.artfess.dataShare.util.DorisUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.StringUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import com.artfess.base.controller.BaseController;
import com.artfess.dataShare.dataResource.ods.model.BizOdsTable;
import com.artfess.dataShare.dataResource.ods.manager.BizOdsTableManager;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.util.List;

/**
 * 数据资源--ODS数据项信息 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author WH
 * @since 2024-11-13
 */
@RestController
@RequestMapping("/biz/dataResource/odsTable/v1/")
@Api(tags = "数据资源--ODS数据项信息")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ_DATASHARE})
public class BizOdsTableController extends BaseController<BizOdsTableManager, BizOdsTable> {
    @Resource
    BizOdsTableManager tableManager;

    @Resource
    BizOdsFieldManager fieldManager;

    @PostMapping("/importFile")
    @ApiOperation(value = "导入数据资产-ODS数据项", httpMethod = "POST", notes = "导入数据资产-ODS数据项")
    public CommonResult export(@RequestParam(value = "file", required = true) MultipartFile file,String catalogId) throws Exception {
        this.tableManager.export(file,catalogId);
        return new CommonResult<>("导入主表成功");
    }

    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name = "model", value = "实体信息") @Validated({AddGroup.class}) @RequestBody BizOdsTable t) {
        if(StringUtil.isEmpty(t.getId())){
            t.setDeployed(0);
            t.setIsCreateTable(0);
            t.setIsDele("0");
            t.setTableNameEn("ods_"+t.getTableNameEn().replaceFirst("ods_",""));
        }
        boolean result = baseService.save(t);
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<>();
    }

    @PostMapping("/saveTable")
    @ApiOperation(value = "从交换库里面选择表，把模型和字段创建到ODS库", httpMethod = "POST", notes = "从交换库里面选择表，把模型和字段创建到ODS库")
    public CommonResult saveTable(@RequestBody CatalogTableVo vo) throws Exception {
        this.tableManager.saveTable(vo.getCatalogId(),vo.getTableId());
        return new CommonResult<String>(true, "操作成功");
    }

    @PostMapping("/saveTableBatch")
    @ApiOperation("批量创建到ODS库")
    public CommonResult<String> saveTableBatch(@RequestBody List<CatalogTableVo> list) throws Exception {
        boolean result = tableManager.saveTableBatch(list);
        if (!result){
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION);
        }
        return CommonResult.success(null,"操作成功");
    }

    @GetMapping("/detail/{id}")
    @ApiOperation(value = "根据ID查询表信息和字段集合", httpMethod = "GET", notes = "根据ID查询表信息和字段集合")
    public CommonResult queryDetail(@PathVariable String id) throws Exception {
        BizOdsTable table=this.tableManager.getById(id);
        List<BizOdsField> fieldList=fieldManager.queryFieldByTableId(id);
        BizOdsTableVo vo=new BizOdsTableVo();
        BeanUtils.copyNotNullProperties(vo,table);
        vo.setFieldList(fieldList);
        return CommonResult.success(vo,"");
    }

    //发布表
    @PostMapping(value = "/deploy/{tableId}")
    @ApiOperation("未有表发布(默认的建表SQL)-发布表，创建实体表")
    public CommonResult Deploy(@PathVariable String tableId) throws Exception {
        this.tableManager.Deploy(tableId);
        return new CommonResult<String>(true, "操作成功");
    }

    @PostMapping(value = "/queryTableVo/{tableId}")
    @ApiOperation("根据表查询表信息、字段信息、模型")
    public OdsTableDetailVo queryTableVo(@PathVariable String tableId) throws Exception {
        OdsTableDetailVo vo=this.tableManager.queryTableVo(tableId);
        return vo;
    }

    @PostMapping(value = "/queryTableData")
    @ApiOperation("数据查询  分页，加搜索、排序")
    public QueryResultData queryTableData(@RequestBody QueryParamVo queryParamVo) throws Exception {
        return this.tableManager.queryTableData(queryParamVo);
    }

    @PostMapping(value = "/removeTable")
    @ApiOperation("删除表和字段-如果表已经创建同步删除实体表")
    public CommonResult queryTableData(@RequestParam String tableId) throws Exception {
        this.tableManager.removeTable(tableId);
        return new CommonResult<String>(true, "操作成功");
    }

    //导出数据
    @RequestMapping(value = "downloadMainTempFile/{tableId}", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "下载导入的模板", httpMethod = "POST", notes = "下载导入的模板")
    public void downloadMainTempFile(HttpServletResponse response,
                                     @ApiParam(name = "tableId", value = "模板别名", required = true) @PathVariable String tableId) throws Exception {
        baseService.downloadMainTempFile(response, tableId);
    }

    @PostMapping(value = "importMain", produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "导入主表数据", httpMethod = "POST", notes = "导入主表数据")
    public CommonResult<String> importMain(@RequestParam(value = "file", required = true) MultipartFile file,
                                           @RequestParam(value = "tableId", required = true) String tableId) throws Exception {
        baseService.importMain(file, tableId);
        return new CommonResult<>("导入主表数据成功");
    }

  //-------------------------------------------

    @PostMapping(value = "/queryTableSql/{tableId}")
    @ApiOperation("根据表ID获取建表语句")
    public CommonResult<String> queryTableSql(@PathVariable String tableId) throws Exception {
        OdsTableDetailVo vo=this.tableManager.queryTableVo(tableId);
        if(StringUtils.isNotBlank(vo.getTable().getCreateTableSql())){
            return CommonResult.success(vo.getTable().getCreateTableSql(),"获取成功");
        }
        String tableDesc = vo.getTable().getTableDesc() == null ? vo.getTable().getTableNameCh() : vo.getTable().getTableDesc();
        String sql= DorisUtils.getOdsTableSql(vo.getTable().getTableNameEn(),tableDesc,vo.getFieldList());
        return CommonResult.success(sql,"获取成功");
    }

    @PostMapping(value = "checkTableExists", produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "验证表是否已经存在", httpMethod = "POST", notes = "验证表是否已经存在")
    public CommonResult<String> checkTableExists(@ApiParam(name = "tableNameEn", value = "表英文名") @RequestParam String tableNameEn,
                                                 @ApiParam(name = "type", value = "发布类型(0:未有表发布，1:已有表发布)") @RequestParam(required = false) String type) throws Exception {
        boolean isExists=tableManager.checkTableExists(tableNameEn);
        if ("1".equals(type)){
            if(isExists){
                return new CommonResult<>(true,"验证成功，可以发布");
            }
            return new CommonResult<>(false,"表"+tableNameEn+"不存在！");
        }
        if(isExists){
            return new CommonResult<>(false,"表"+tableNameEn+"已经存在，请更换表名");
        }
        return new CommonResult<>(true,"可以使用表名");
    }

    @PostMapping(value = "checkPhysicsTableExists", produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "验证物理表是否已经存在", httpMethod = "POST", notes = "验证表是否已经存在")
    public CommonResult<String> checkPhysicsTableExists(@ApiParam(name = "tableNameEn", value = "表英文名") @RequestParam String tableNameEn,
                                                        @ApiParam(name = "type", value = "发布类型(0:未有表发布，1:已有表发布)") @RequestParam(required = false) String type) throws Exception {
        boolean isExists=tableManager.checkPhysicsTableExists(tableNameEn);
        if ("1".equals(type)){
            if(isExists){
                return new CommonResult<>(true,"验证成功，可以发布");
            }
            return new CommonResult<>(false,"表"+tableNameEn+"不存在！");
        }
        if(isExists){
            return new CommonResult<>(false,"表"+tableNameEn+"已经存在，请更换表名");
        }
        return new CommonResult<>(true,"可以使用表名");
    }

    @PostMapping(value = "saveTableSql", produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "保存建表SQL", httpMethod = "POST", notes = "保存建表SQL")
    public CommonResult<String> saveTableSql(@ApiParam(name = "tableId", value = "数据表ID") @RequestParam String tableId,
                                             @ApiParam(name = "tableSql", value = "建表语句") @RequestParam String tableSql) throws Exception {
        BizOdsTable table=tableManager.getById(tableId);
        table.setCreateTableSql(tableSql);
        tableManager.updateById(table);
        return new CommonResult<>(true,"保存建表SQL成功");
    }

    @RequestMapping(value = "deployHaveTable", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "已有表发布", httpMethod = "POST", notes = "已有表发布")
    public CommonResult<String> deployHaveTable(@ApiParam(name = "tableId", value = "数据表ID") @RequestParam String tableId) throws Exception {
        tableManager.deployHaveTable(tableId);
        return new CommonResult<>(true,"已有表发布成功");
    }

    @RequestMapping(value = "deployDotHaveTable", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "未有表发布", httpMethod = "POST", notes = "未有表发布")
    public CommonResult<String> deployDotHaveTable(@ApiParam(name = "tableId", value = "数据表ID") @RequestParam String tableId,
                                                   @ApiParam(name = "tableSql", value = "建表语句") @RequestParam String tableSql) throws Exception {
        tableManager.deployDotHaveTable(tableId,tableSql);
        return new CommonResult<>(true,"未有表发布");
    }

    @RequestMapping(value = "updateCatalog", method = RequestMethod.PUT, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "修改表资源目录", httpMethod = "PUT", notes = "修改表资源目录")
    public CommonResult<String> updateCatalog(@RequestParam String[] ids,@RequestParam String catalogId){
        boolean result = tableManager.updateCatalog(ids,catalogId);
        if (result){
            return new CommonResult<>(true,"操作成功");
        }
        return new CommonResult<>(false,"操作失败");
    }
}
