package com.artfess.uc.manager.impl;

import com.artfess.base.context.BaseContext;
import com.artfess.base.model.CommonResult;
import com.artfess.base.util.BeanUtils;
import com.artfess.uc.dto.UserApprovalRecordDTO;
import com.artfess.uc.exception.RequiredException;
import com.artfess.uc.manager.OrgPostManager;
import com.artfess.uc.manager.UserManager;
import com.artfess.uc.model.BizUserApprovalRecord;
import com.artfess.uc.dao.BizUserApprovalRecordDao;
import com.artfess.uc.manager.BizUserApprovalRecordManager;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.uc.model.User;
import com.artfess.uc.vo.UserApprovalRecordVO;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.lang.reflect.InvocationTargetException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 用户注册审核记录表 服务实现类
 *
 * @author 系统管理员
 * @company 阿特菲斯信息技术有限公司
 * @since 2025-02-11
 */
@Service
public class BizUserApprovalRecordManagerImpl extends BaseManagerImpl<BizUserApprovalRecordDao, BizUserApprovalRecord> implements BizUserApprovalRecordManager {

    @Resource
    private BaseContext baseContext;
    @Resource
    private UserManager userManager;
    @Resource
    private OrgPostManager orgPostService;


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void todo(UserApprovalRecordDTO dto) {
        if (!isPost(dto.getUserId())) {
            throw new RequiredException("提交失败，用户岗位信息为空，无法提交！");
        }
        // 先发起一条记录
        BizUserApprovalRecord applyStart = new BizUserApprovalRecord();
        try {
            BeanUtils.copyNotNullProperties(applyStart,dto);
        } catch (IllegalAccessException | InvocationTargetException e) {
            throw new RuntimeException(e);
        }
        applyStart.setApprovalUserId(baseContext.getCurrentUserId());
        applyStart.setApprovalUserName(baseContext.getCurrentUserName());
        applyStart.setApprovalUserAccount(baseContext.getCurrentUserAccout());
        switch (dto.getTodoType()) {
            case "start_r":
                applyStart.setApprovalNode("提交申请");
                applyStart.setApprovalResult(3);
                applyStart.setProcessingTime(0L);
                break;
            case "start":
                // 同意后修改用户状态
                updateUserStatus(dto.getUserId(), 3);
                applyStart.setApprovalNode("提交申请");
                applyStart.setApprovalResult(3);
                applyStart.setIntranetPort("重新发起审批");
                applyStart.setProcessingTime(getProcessingTime(dto.getUserId()));
                break;
            case "agree":
                // 同意后修改用户状态
                updateUserStatus(dto.getUserId(), 1);
                applyStart.setApprovalNode("管理员审核");
                applyStart.setApprovalResult(1);
                // 处理时长S
                applyStart.setProcessingTime(getProcessingTime(dto.getUserId()));
                break;
            case "reject":
                // 同意后修改用户状态
                updateUserStatus(dto.getUserId(), 4);
                applyStart.setApprovalNode("管理员审核");
                applyStart.setApprovalResult(0);
                // 处理时长S
                applyStart.setProcessingTime(getProcessingTime(dto.getUserId()));
                break;
            default:
                throw new RuntimeException("操作类型异常，无法判断！");
        }
        applyStart.setApprovalNode("提交申请");
        applyStart.setApprovalOrgId(baseContext.getCurrentOrgId());
        applyStart.setApprovalOrgName(baseContext.getCurrentOrgName());
        applyStart.setApprovalDeptId(baseContext.getCurrentOrgName());
        applyStart.setApprovalDeptName(baseContext.getCurrentDeptId());
        applyStart.setApprovalDate(LocalDateTime.now());
        this.save(applyStart);
    }

    @Override
    public CommonResult<List<UserApprovalRecordVO>> getRecordByUserId(String userId) {
        Assert.hasText(userId, "用户ID不能为空！");
        List<UserApprovalRecordVO> vo = new ArrayList<>();

        LambdaQueryWrapper<BizUserApprovalRecord> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(BizUserApprovalRecord::getUserId, userId);
        queryWrapper.orderByAsc(BizUserApprovalRecord::getCreateTime);
        final List<BizUserApprovalRecord> list = this.list(queryWrapper);
        if (CollectionUtils.isNotEmpty(list)) {
            list.forEach(e -> {
                UserApprovalRecordVO applyRecordVo = new UserApprovalRecordVO();
                try {
                    BeanUtils.copyNotNullProperties(applyRecordVo,e);
                } catch (IllegalAccessException | InvocationTargetException ex) {
                    throw new RuntimeException(ex);
                }
                //BeanUtils.copyProperties(e, applyRecordVo);
                vo.add(applyRecordVo);
            });
        }

        return new CommonResult<>(true, "查询成功", vo);
    }

    private long getProcessingTime(String userId) {
        LambdaQueryWrapper<BizUserApprovalRecord> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(BizUserApprovalRecord::getUserId, userId);
        queryWrapper.orderByDesc(BizUserApprovalRecord::getCreateTime);
        queryWrapper.last("limit 1");
        List<BizUserApprovalRecord> list = this.list(queryWrapper);
        if (CollectionUtils.isNotEmpty(list)) {
            LocalDateTime lastTime = list.get(0).getApprovalDate();
            // 计算两个时间点之间的持续时间
            Duration duration = Duration.between(lastTime, LocalDateTime.now());
            // 获取相差的秒数
            return duration.getSeconds();
        }
        return 0;
    }

    /**
     * 修改用户状态
     *
     * @param userId 用户ID
     * @param status 状态 0:禁用，1正常，-1未激活，-2离职, 3待审核 ,4待发起
     */
    private void updateUserStatus(String userId, Integer status) {
        User user = userManager.getById(userId);
        user.setStatus(status);
        userManager.updateById(user);
    }

    /**
     * 判断当前用户是否存在岗位
     *
     * @param userId 用户ID
     */
    private boolean isPost(String userId) {
        List<Map<String, Object>> orgPostList = orgPostService.getUserByUserId(userId);
        if (CollectionUtils.isNotEmpty(orgPostList)) {
            Map<String, Object> postMap = orgPostList.get(0);
            if (postMap.containsKey("postId")) {
                Object postId = postMap.get("postId");
                return BeanUtils.isNotEmpty(postId);
            }
            return false;
        }
        return false;
    }

}
