import { describe, it, expect, vi, beforeEach } from 'vitest'
import { setActivePinia, createPinia } from 'pinia'
import { useMenuStore } from '../menu'

// Mock API
vi.mock('@/api/system/menu', () => ({
    userMenuQuery: vi.fn()
}))

describe('菜单Store测试', () => {
    beforeEach(() => {
        setActivePinia(createPinia())
    })

    it('初始状态正确', () => {
        const store = useMenuStore()
        
        expect(store.userMenu).toEqual([])
        expect(store.normalMenu).toEqual([])
        expect(store.currentMenuGroup).toBe(null)
        expect(store.currentMenuGroupAlias).toBe('')
        expect(store.customMenu).toEqual([])
    })

    it('设置菜单数据', () => {
        const store = useMenuStore()
        const mockMenu = [
            { id: '1', name: '菜单1', alias: 'menu1' },
            { id: '2', name: '菜单2', alias: 'menu2' }
        ]
        
        store.setMenu(mockMenu)
        
        expect(store.userMenu).toEqual(mockMenu)
    })

    it('设置当前菜单组', () => {
        const store = useMenuStore()
        const mockMenuGroup = { id: '1', name: '测试菜单', alias: 'test' }
        
        store.setCurrentMenuGroup(mockMenuGroup)
        store.setCurrentMenuGroupAlias('test')
        
        expect(store.currentMenuGroup).toEqual(mockMenuGroup)
        expect(store.currentMenuGroupAlias).toBe('test')
    })

    it('清除菜单数据', () => {
        const store = useMenuStore()
        
        // 设置初始数据
        store.setMenu([{ id: '1', name: '测试' }])
        store.setCurrentMenuGroupAlias('test')
        
        store.clearMenu()
        
        expect(store.userMenu).toEqual([])
        expect(store.normalMenu).toEqual([])
        expect(store.currentMenuGroup).toBe(null)
        expect(store.currentMenuGroupAlias).toBe('')
        expect(store.customMenu).toEqual([])
    })

    it('获取菜单数据成功', async () => {
        const { userMenuQuery } = await import('@/api/system/menu')
        
        const mockResponse = [
            {
                id: '1',
                name: '系统管理',
                alias: 'normal',
                children: [
                    { id: '11', name: '用户管理', alias: 'user' }
                ]
            }
        ]
        
        vi.mocked(userMenuQuery).mockResolvedValue(mockResponse)
        
        const store = useMenuStore()
        store.fixFirstMenu = false // 设置为同步模式
        const result = await store.userMenuData()
        
        expect(userMenuQuery).toHaveBeenCalled()
        expect(typeof result).toBe('string') // 返回路径字符串
        expect(store.userMenu.length).toBeGreaterThan(0)
    })

    it('处理空菜单响应', async () => {
        const { userMenuQuery } = await import('@/api/system/menu')
        
        vi.mocked(userMenuQuery).mockResolvedValue([])
        
        const store = useMenuStore()
        store.fixFirstMenu = false // 设置为同步模式
        const result = await store.userMenuData()
        
        expect(result).toBe('') // 空响应返回空字符串
        expect(store.userMenu).toEqual([])
    })

    it('记录和获取路由参数', () => {
        const store = useMenuStore()
        const routeName = 'testRoute'
        const params = { id: '123' }
        const query = { tab: 'detail' }
        
        // 记录路由参数
        store.recordRouteParams(routeName, params, query)
        
        // 获取路由参数
        const storedParams = store.getRouteParams(routeName)
        
        expect(storedParams).toEqual({ params, query })
    })

    it('清除路由参数', () => {
        const store = useMenuStore()
        const routeName = 'testRoute'
        
        // 先记录参数
        store.recordRouteParams(routeName, { id: '123' }, { tab: 'detail' })
        expect(store.getRouteParams(routeName)).not.toBe(null)
        
        // 清除参数
        store.clearRouteParams(routeName)
        expect(store.getRouteParams(routeName)).toBe(null)
    })

    it('清除菜单时同时清除路由参数', () => {
        const store = useMenuStore()
        
        // 设置初始数据
        store.setMenu([{ id: '1', name: '测试' }])
        store.recordRouteParams('testRoute', { id: '123' }, { tab: 'detail' })
        
        store.clearMenu()
        
        expect(store.userMenu).toEqual([])
        expect(store.getRouteParams('testRoute')).toBe(null)
        expect(store.routeParamsMap).toEqual({})
    })
})