package com.artfess.xqxt.meeting.controller;


import com.alibaba.fastjson.JSONArray;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.xqxt.meeting.manager.BizMeetingTopicManager;
import com.artfess.xqxt.meeting.manager.BizMeetingVoteManager;
import com.artfess.xqxt.meeting.model.BizMeetingTopic;
import com.artfess.xqxt.meeting.model.BizMeetingVote;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * 会议议题投票表 前端控制器
 *
 * @author 管理员
 * @company 阿特菲斯信息技术有限公司
 * @since 2022-11-03
 */
@RestController
@RequestMapping("/bizMeetingVote/v1")
@Api(value = "会议投票", tags = "会议投票")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class BizMeetingVoteController extends BaseController<BizMeetingVoteManager, BizMeetingVote> {

    @Autowired
    private BizMeetingTopicManager topicManager;

    @ApiOperation(value = "保存投票信息", notes = "兼容新增及修改")
    @PostMapping("/createVote")
    public CommonResult<String> createVote(@RequestBody BizMeetingTopic topic) {
        boolean result = baseService.createVote(topic);
        return new CommonResult<>(result, "操作" + (result ? "成功" : "失败"), topic.getId());
    }

    @ApiOperation(value = "用户投票")
    @PostMapping("/vote")
    @Transactional
    public CommonResult<String> vote(@RequestBody List<BizMeetingVote> vote) {
        boolean result = false;
        for (BizMeetingVote v : vote) {
            result = baseService.vote(v);
            if (!result) {
                throw new RuntimeException("操作失败，回滚事务......");
            }
        }
        return new CommonResult<>(result, "操作" + (result ? "成功" : "失败"));
    }

    @ApiOperation(value = "修改投票状态", notes = "使用字典（1：已保存，2：开启投票，3：关闭投票，4：已发布投票结果）")
    @PostMapping("/changeVoteStatus")
    public CommonResult<String> changeVoteStatus(@RequestBody BizMeetingTopic topic) {
        boolean result = baseService.changeVoteStatus(topic);
        return new CommonResult<>(result, "操作" + (result ? "成功" : "失败"));
    }

    @GetMapping("/detail/{id}")
    @ApiOperation("投票详情·ID查询")
    public BizMeetingTopic getDetail(@ApiParam(name = "id", value = "实体id") @PathVariable String id) {
        return baseService. detail(id);
    }

    @PostMapping(value = "/queryVote")
    @ApiOperation("投票详情·分页")
    public PageList<BizMeetingTopic> queryVote(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<BizMeetingTopic> queryFilter) {
        return baseService.selectByUserId(queryFilter);
    }

    @PostMapping("/checkUserVote")
    @ApiOperation(value = "检查当前用户是否已经投过票", notes = "key 为问题ID，value 为 true 表示当前登录用户已经为该问题投过票了")
    public CommonResult<Map<String, Boolean>> checkUserVote(@RequestParam String topicId) {
        Map<String, Boolean> result = baseService.checkUserVote(topicId);
        return new CommonResult<>(true, "", result);
    }

    @DeleteMapping("/deleteVoteInfoByTopicIds")
    @ApiOperation("根据topicId批量删除投票信息")
    public CommonResult<String> deleteVoteInfoByTopicIds(@ApiParam(name = "topicIds", value = "topicIds") @RequestParam String... topicIds) {
        boolean result = topicManager.removeByIds(Arrays.asList(topicIds));
        return new CommonResult<>(result, "操作" + (result ? "成功" : "失败"));
    }

    @PostMapping("/exportDataToExcel")
    @ApiOperation(value = "导出数据到Excel")
    public CommonResult<String> exportDataToExcel(@ApiParam(name = "queryFilter", value = "topic查询器") @RequestBody QueryFilter<BizMeetingTopic> queryFilter, HttpServletResponse response) throws IOException {
        baseService.exportDataToExcel(queryFilter, response);
        return new CommonResult<>("导出成功");
    }

    @PostMapping(value = "/getVoteResult")
    @ApiOperation(value = "投票结果", notes = "传入会议ID")
    public JSONArray getVoteResult(@ApiParam(name = "meetingId", value = "会议ID") @RequestParam String meetingId) {
        return baseService.getVoteResult(meetingId);
    }

    @PostMapping(value = "/votedCount")
    @ApiOperation(value = "投票统计", notes = "传入topicId")
    public CommonResult<Integer> votedCount(@ApiParam(name = "topicId", value = "topicId") @RequestParam String topicId) {
        return new CommonResult<>(true, "操作成功", baseService.votedCount(topicId));
    }

    @PostMapping(value = "/checkTheme")
    @ApiOperation(value = "投票主题验证")
    public CommonResult<Integer> checkTheme(@ApiParam(name = "meetingId", value = "meetingId") @RequestParam String meetingId,
                                            @ApiParam(name = "theme", value = "theme") @RequestParam String theme) {
        if (baseService.checkTheme(meetingId,theme)>0){
            return new CommonResult<>(false, "主题已存在");
        }
        return new CommonResult<>(true, "操作成功");
    }





}
