package com.artfess.application.persistence.manager.impl;

import com.alibaba.fastjson.JSONObject;
import com.artfess.application.model.MessageConfig;
import com.artfess.application.model.MsgTemplate;
import com.artfess.application.model.RemindConfig;
import com.artfess.application.persistence.dao.RemindConfigDao;
import com.artfess.application.persistence.manager.MessageConfigManager;
import com.artfess.application.persistence.manager.MsgTemplateManager;
import com.artfess.application.persistence.manager.RemindConfigManager;
import com.artfess.base.context.BaseContext;
import com.artfess.base.exception.RequiredException;
import com.artfess.base.feign.UCFeignService;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageBean;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.JsonUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.job.persistence.manager.SchedulerService;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.gexin.fastjson.JSON;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.quartz.JobKey;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import org.quartz.TriggerKey;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.List;
import java.util.Map;

/**
 * 业务提醒配置 服务实现类
 *
 * @author min.wu
 * @company 阿特菲斯信息技术有限公司
 * @since 2023-12-21
 */
@Service
public class RemindConfigManagerImpl extends BaseManagerImpl<RemindConfigDao, RemindConfig> implements RemindConfigManager {

    @Autowired
    SchedulerService schedulerService;

    @Autowired
    Scheduler scheduler;

    @Autowired
    private MessageConfigManager messageConfigManager;

    @Autowired
    private MsgTemplateManager msgTemplateManager;

    @Resource
    UCFeignService ucFeignService;

    @Resource
    BaseContext baseContext;

    @Override
    public PageList<RemindConfig> findByPage(QueryFilter<RemindConfig> queryFilter) {
        PageBean pageBean = queryFilter.getPageBean();
        Class<RemindConfig> currentModelClass = currentModelClass();
        IPage<RemindConfig> result = baseMapper.selectPage(convert2IPage(pageBean), convert2Wrapper(queryFilter, currentModelClass));
        result.getRecords().forEach(remindConfig -> {
            try {
                String jobName = remindConfig.getName() + ":" + remindConfig.getCode() + ":" + scheduler.getSchedulerName();
                TriggerKey key = new TriggerKey(jobName, baseContext.getCurrentTenantId());
                Trigger.TriggerState state = scheduler.getTriggerState(key);
                remindConfig.setTaskStatus(state.name());
            } catch (SchedulerException e) {
                e.printStackTrace();
            }

        });
        return new PageList<RemindConfig>(result);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean saveOrUpdateInfo(RemindConfig remindConfig) throws Exception {
        Assert.hasText(remindConfig.getTplKey(), "请选择短信模板");
        if (checkSameCode(remindConfig.getCode(), remindConfig.getId())) {
            throw new RequiredException("添加工作提醒任务失败，该任务【" + remindConfig.getCode() + "】在系统中已存在，不能重复！");
        }

        String jobName = remindConfig.getName() + ":" + remindConfig.getCode() + ":" + scheduler.getSchedulerName();
        boolean jobExists = schedulerService.isJobExists(jobName);

        boolean isSuccess = true;
        if (!jobExists) {
            String className = remindConfig.getPlanClass();
            String description = remindConfig.getDescription();
            Map<String, Object> parameter = Maps.newHashMap();
            parameter.put("msgId", remindConfig.getMsgId());
            CommonResult<String> result = schedulerService.addJob(jobName, className, parameter, description);
            isSuccess = result.getState();
        }
        if (isSuccess) {
            boolean b = this.saveOrUpdate(remindConfig);
            if (b) {
                processMessageConfig(remindConfig);
            }
            return b;

        } else {
            return isSuccess;
        }
    }

    private void processMessageConfig(RemindConfig remindConfig) throws IOException {
        QueryWrapper<MessageConfig> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("TPL_KEY_", remindConfig.getTplKey());
        List<MessageConfig> messageConfigs = messageConfigManager.list(queryWrapper);
        MessageConfig messageConfig = new MessageConfig();
        if (!messageConfigs.isEmpty()) {
            messageConfig = messageConfigs.get(0);
        }

        messageConfig.setTplKey(remindConfig.getTplKey());
        messageConfig.setCode(remindConfig.getCode());
        messageConfig.setName(remindConfig.getName());
        messageConfig.setNotifyType(remindConfig.getNotifyType());
        messageConfig.setUserStr(remindConfig.getUserStr());
        messageConfig.setUserConfigType(remindConfig.getUserConfigType());
        String userStr = remindConfig.getUserStr();
        Assert.hasText(userStr, "请选择接收人");
        List<Map<String, Object>> list = JsonUtil.toBean(userStr, List.class);
        String userId = "";
        String orgId = "";
        for (Map<String, Object> m : list) {
            String type = String.valueOf(m.get("type"));
            if ("user".equalsIgnoreCase(type)) {
                String codes = String.valueOf(m.get("codes"));
                if (StringUtils.isBlank(codes)) {
                    continue;
                }
                ArrayNode userByAccounts = ucFeignService.getUserByAccounts(codes);
                for (JsonNode userByAccount : userByAccounts) {
                    String id = userByAccount.get("id").toString();
                    if (StringUtils.isNotBlank(id)) {
                        userId = id.replace("\"", "").replace("\"", "") + ",";
                    }
                }
            }
            if ("org".equalsIgnoreCase(type)) {
                String codes = String.valueOf(m.get("codes"));
                if (StringUtils.isBlank(codes)) {
                    continue;
                }
                ArrayNode orgList = this.ucFeignService.getOrgListByCodes(codes);
                for (JsonNode org : orgList) {
                    orgId = org.get("id").toString() + ",";
                }
            }
            if (!StringUtil.isEmpty(orgId)) {
                messageConfig.setOrgId(orgId.substring(0, orgId.lastIndexOf(",")));
            }
            if (!StringUtil.isEmpty(userId)) {
                messageConfig.setUserId(userId.substring(0, userId.lastIndexOf(",")));
            }
            messageConfigManager.saveOrUpdate(messageConfig);
        }
    }

    private boolean checkSameCode(String code, String id) {
        Assert.hasText(code, "工作提醒编码不能为空。");
        QueryWrapper<RemindConfig> queryWrapper = new QueryWrapper<RemindConfig>();
        queryWrapper.eq("code_", code);
        queryWrapper.ne(StringUtil.isNotEmpty(id), "ID_", id);
        int num = this.baseMapper.selectCount(queryWrapper);
        if (num != 0) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean deleteRemindConfigById(String id) throws SchedulerException {

        if (null == scheduler) {
            return false;
        }
        Assert.hasText(id, "请选择要操作的工作提醒");
        RemindConfig remindConfig = baseMapper.selectById(id);
        Assert.notNull(remindConfig, "工作提醒不存在");
        boolean deleteStatus = this.removeById(id);
        if (!deleteStatus) {
            return deleteStatus;
        }

        String jobName = remindConfig.getName() + ":" + remindConfig.getCode() + ":" + scheduler.getSchedulerName();
        //判断是否存在定时任务的触发器
        boolean rtn = schedulerService.isTriggerExists(jobName);
        if (rtn) {
            schedulerService.delTrigger(jobName);
        }
        boolean isExist = schedulerService.isJobExists(jobName);
        if (isExist) {
            schedulerService.delJob(jobName);
        }

        return true;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean toggleTriggerRun(String id) throws Exception {
        if (null == scheduler) {
            return false;
        }
        Assert.hasText(id, "请选择要操作的工作提醒");
        RemindConfig remindConfig = baseMapper.selectById(id);
        Assert.notNull(remindConfig, "工作提醒不存在");
        String jobName = remindConfig.getName() + ":" + remindConfig.getCode() + ":" + scheduler.getSchedulerName();
        TriggerKey key = new TriggerKey(jobName, baseContext.getCurrentTenantId());
        Trigger.TriggerState state = scheduler.getTriggerState(key);
        //判断是否存在定时任务的触发器
        boolean rtn = schedulerService.isTriggerExists(jobName);
        if (!rtn) {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("type", remindConfig.getPlanType());
            jsonObject.put("timeInterval", remindConfig.getPlanInterval());
            schedulerService.addTrigger(jobName, jobName, JSON.toJSONString(jsonObject));
        } else {
            if (state == Trigger.TriggerState.PAUSED) {
                //启动定时器
                scheduler.resumeTrigger(key);
            } else if (state == Trigger.TriggerState.NORMAL) {
                scheduler.pauseTrigger(key);
            }
        }
        return true;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean executeJob(String id) throws SchedulerException {

        if (null == scheduler) {
            return false;
        }
        Assert.hasText(id, "请选择要操作的工作提醒");
        RemindConfig remindConfig = baseMapper.selectById(id);

        String jobName = remindConfig.getName() + ":" + remindConfig.getCode() + ":" + scheduler.getSchedulerName();
        JobKey key = new JobKey(jobName, baseContext.getCurrentTenantId());
        scheduler.triggerJob(key);
        return true;
    }

    @Override
    public RemindConfig findById(String id) {
        Assert.hasText(id, "请选择要操作的工作提醒");
        RemindConfig remindConfig = baseMapper.selectById(id);
        Assert.notNull(remindConfig, "工作提醒不存在");
        MsgTemplate msgTemplate = msgTemplateManager.get(remindConfig.getMsgId());
        if(null != msgTemplate) {
            QueryWrapper<MessageConfig> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("TPL_KEY_", msgTemplate.getKey());
            List<MessageConfig> messageConfigs = messageConfigManager.list(queryWrapper);
            MessageConfig messageConfig = new MessageConfig();
            if (!messageConfigs.isEmpty()) {
                messageConfig = messageConfigs.get(0);
                remindConfig.setTplKey(messageConfig.getTplKey());
                remindConfig.setNotifyType(messageConfig.getNotifyType());
                remindConfig.setUserStr(messageConfig.getUserStr());
            }
        }

        return remindConfig;
    }
}
