/*
 * ✨ 请求封装器工具类函数 ✨
 * */

// 类型定义
interface PageBean {
    pageSize: number
    page?: number
    total?: number
}

interface QueryItem {
    property: string
    operation?: string
    value?: any
    defaultValue?: any
    group?: string
    relation?: string
}

interface SorterItem {
    property: string
    direction?: string
}

interface RequestBody {
    pageBean?: PageBean
    querys?: QueryItem[]
    sorter?: SorterItem[]
}

interface SorterResult {
    prop?: string
    order?: string
}

/**
 * 检查并处理分页请求体
 * @param body 分页请求对象
 * @returns 处理后的分页请求配置对象
 */
export function refineRequestBody(body: RequestBody = {}): RequestBody {
    // 如果请求体为空对象，则返回空对象
    if (!Object.keys(body).length) return { pageBean: { pageSize: -1 } }

    // 辅助函数：判断值是否为空（空字符串、空数组、null 或 undefined）
    const _isEmpty = (value: any): boolean =>
        value === '' ||
        (Array.isArray(value) && !value.length) ||
        value === null ||
        value === undefined

    const { pageBean = { pageSize: -1 }, querys = [], sorter = [] } = body

    return {
        pageBean,
        querys: querys.flatMap((item) => {
            const { defaultValue, ...i } = item
            let value = item.value

            if (_isEmpty(value)) {
                // 如果 value 为空，则检查 defaultValue
                value = defaultValue
                // 如果 defaultValue 也为空，则移除该查询对象
                if (_isEmpty(value)) return []
            }

            // 如果为LIKE查询时，处理为字符串格式
            if (i.operation === 'LIKE' && Array.isArray(value)) {
                value = value.join()
            }

            // 返回处理后的查询对象，加入默认属性
            return {
                group: 'main',
                operation: 'EQUAL',
                relation: 'AND',
                ...i,
                value
            }
        }),
        // 过滤掉没有 direction 属性的排序对象
        sorter: sorter.filter((i) => i.direction)
    }
}

/**
 * 用于更改后端分页查询配置项中的查询字段值
 * @param params 分页请求对象
 * @param props 字段名
 * @param value 字段值
 * @param operation 操作类型
 */
export function setQueryValue(
    params: RequestBody = {},
    props: string = '',
    value: any = '',
    operation: string = ''
): void {
    params?.querys?.forEach((i) => {
        if (i.property.trim() !== props.trim()) return
        i.value = value
        if (operation) {
            i.operation = operation
        }
    })
}

/**
 * 用于查询后端分页查询配置项中的字段值
 * @param params 分页请求对象
 * @param props 字段名
 * @returns 字段值
 */
export function getQueryValue(params: RequestBody = {}, props: string = ''): any {
    const queryItem = params?.querys?.find((i) => i.property.trim() === props.trim())
    if (!queryItem) return undefined

    const { value, defaultValue } = queryItem

    // 辅助函数：判断值是否为空（空字符串、空数组、null 或 undefined）
    const isEmpty = (value: any): boolean =>
        value === '' ||
        (Array.isArray(value) && !value.length) ||
        value === null ||
        value === undefined

    let result: any
    if (isEmpty(value)) {
        if (!isEmpty(defaultValue)) result = defaultValue
    } else {
        result = value
    }
    return result
}

/**
 * 用于查询后端分页查询配置项中的query绑定元素对象
 * @param params 分页请求对象
 * @param props 字段名
 * @returns query绑定元素对象
 */
export function getQueryItem(params: RequestBody = {}, props: string = ''): QueryItem {
    const index = params?.querys?.findIndex((i) => i.property.trim() === props.trim())
    return index !== -1 ? params?.querys![index] : ({} as QueryItem)
}

/**
 * 用于更改后端分页排序配置项中的排序规则
 * @param params 分页请求对象
 * @param props 字段名
 * @param direction 排序方式
 */
export function setSorterDirection(
    params: RequestBody = {},
    props: string = '',
    direction: string = ''
): void {
    params?.sorter?.forEach((i) => {
        if (i.property.trim() !== props.trim()) {
            i.direction = ''
        } else {
            i.direction = direction
        }
    })
}

/**
 * 用于查询后端分页排序配置项中的排序规则
 * @param params 分页请求对象
 * @param isNotFullName 排序方式规则字符串是否全称
 * @returns sorter绑定排序对象
 */
export function getSorterDirection(
    params: RequestBody = {},
    isNotFullName: boolean = false
): SorterResult {
    const { sorter = [] } = params || {}
    if (!(Array.isArray(sorter) && sorter.length)) return {}

    const sorterItem = sorter.find((i) => i.direction)
    if (!sorterItem) return {}

    const { property: prop = '', direction: order = '' } = sorterItem
    if (!prop) return {}

    const orderOpt: Record<string, string> = { ASC: 'ascending', DESC: 'descending' }
    return { prop, order: isNotFullName ? orderOpt[order] || '' : order }
}
