package com.artfess.portal.kh.controller;


import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.validgroup.UpdateGroup;
import com.artfess.portal.kh.manager.KhResultManager;
import com.artfess.portal.kh.manager.KhSchemePlanManager;
import com.artfess.portal.kh.model.KhResult;
import com.artfess.portal.kh.model.KhSchemePlan;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Assert;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDate;
import java.util.Arrays;

/**
 * 考核计划信息 前端控制器
 *
 * @author min.wu
 * @company 阿特菲斯信息技术有限公司
 * @since 2024-02-20
 */
@Slf4j
@RestController
@Api(tags = "绩效考核-考核计划信息")
@RequestMapping("/kh/scheme/plan/")
@ApiGroup(group = {ApiGroupConsts.GROUP_SYSTEM})
public class KhSchemePlanController extends BaseController<KhSchemePlanManager, KhSchemePlan> {

    @Autowired
    private KhResultManager resultManager;

    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({AddGroup.class}) KhSchemePlan t) {
        boolean result = baseService.createInfo(t);
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<String>();
    }

    @Override
    @PutMapping("/")
    @ApiOperation("更新实体")
    public CommonResult<String> updateById(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({UpdateGroup.class}) KhSchemePlan t) {
        boolean result = baseService.updateInfo(t);
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<String>();
    }

    @Override
    @GetMapping("/{id}")
    @ApiOperation("根据id查询实体")
    public KhSchemePlan getById(@ApiParam(name = "id", value = "实体id") @PathVariable String id) {
        return baseService.findById(id);
    }

    @GetMapping("/executePlan")
    @ApiOperation("执行计划")
    public CommonResult<String> executePlan(@ApiParam(name = "ids", value = "实体集合") @RequestParam String... ids) {
        Assert.notNull(ids, "请选择要执行的计划！");
        boolean result = baseService.updateStatus(Arrays.asList(ids), "execute");
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "执行计划失败");
        }
        result = resultManager.createResult((Arrays.asList(ids)).get(0));
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "执行计划失败");
        }
        return new CommonResult<>();
    }

    @GetMapping("/pausePlan")
    @ApiOperation("暂停计划")
    public CommonResult<String> pausePlan(@ApiParam(name = "ids", value = "实体集合") @RequestParam String... ids) {
        boolean result = baseService.updateStatus(Arrays.asList(ids), "stop");
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "暂停计划失败");
        }
        return new CommonResult<>();
    }

    @GetMapping("/endPlan")
    @ApiOperation("结束计划")
    public CommonResult<String> endPlan(@ApiParam(name = "ids", value = "实体集合") @RequestParam String... ids) {
        boolean result = baseService.updateStatus(Arrays.asList(ids), "end");
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "结束计划失败");
        }
        return new CommonResult<>();
    }

    @Override
    @PostMapping(value = "/query", produces = {"application/json; charset=utf-8"})
    @ApiOperation("分页查询结果")
    public PageList<KhSchemePlan> query(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<KhSchemePlan> queryFilter) {
        PageList<KhSchemePlan> query = baseService.query(queryFilter);
        query.getRows().forEach(plan -> {
            if(plan.getEndDate().isBefore(LocalDate.now())) {
                plan.setStatus("end");
            }
        });
        return query;
    }
}
