/*
 * Decompiled with CFR 0.152.
 */
package org.jetlinks.community.network.mqtt.server.vertx;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.handler.codec.DecoderException;
import io.netty.handler.codec.mqtt.MqttConnectReturnCode;
import io.netty.handler.codec.mqtt.MqttProperties;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.util.ReferenceCountUtil;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.net.SocketAddress;
import io.vertx.mqtt.MqttEndpoint;
import io.vertx.mqtt.MqttTopicSubscription;
import io.vertx.mqtt.messages.MqttPublishMessage;
import io.vertx.mqtt.messages.MqttSubscribeMessage;
import io.vertx.mqtt.messages.MqttUnsubscribeMessage;
import java.net.InetSocketAddress;
import java.time.Duration;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import org.jetlinks.community.network.mqtt.server.MqttConnection;
import org.jetlinks.community.network.mqtt.server.MqttPublishing;
import org.jetlinks.community.network.mqtt.server.MqttSubscription;
import org.jetlinks.community.network.mqtt.server.MqttUnSubscription;
import org.jetlinks.core.message.codec.EncodedMessage;
import org.jetlinks.core.message.codec.MqttMessage;
import org.jetlinks.core.message.codec.SimpleMqttMessage;
import org.jetlinks.core.server.mqtt.MqttAuth;
import org.jetlinks.core.utils.Reactors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.publisher.Sinks;

class VertxMqttConnection
implements MqttConnection {
    private static final Logger log = LoggerFactory.getLogger(VertxMqttConnection.class);
    private final MqttEndpoint endpoint;
    private long keepAliveTimeoutMs;
    private long lastPingTime = System.currentTimeMillis();
    private volatile boolean closed = false;
    private volatile boolean accepted = false;
    private volatile boolean autoAckSub = true;
    private volatile boolean autoAckUnSub = true;
    private volatile boolean autoAckMsg = false;
    private int messageIdCounter;
    private static final MqttAuth emptyAuth = new MqttAuth(){

        public String getUsername() {
            return "";
        }

        public String getPassword() {
            return "";
        }
    };
    private final Sinks.Many<MqttPublishing> messageProcessor = Reactors.createMany((int)Integer.MAX_VALUE, (boolean)false);
    private final Sinks.Many<MqttSubscription> subscription = Reactors.createMany((int)Integer.MAX_VALUE, (boolean)false);
    private final Sinks.Many<MqttUnSubscription> unsubscription = Reactors.createMany((int)Integer.MAX_VALUE, (boolean)false);
    private final Consumer<MqttConnection> defaultListener = mqttConnection -> {
        log.debug("mqtt client [{}] disconnected", (Object)this.getClientId());
        this.subscription.tryEmitComplete();
        this.unsubscription.tryEmitComplete();
        this.messageProcessor.tryEmitComplete();
    };
    private Consumer<MqttConnection> disconnectConsumer = this.defaultListener;
    private volatile InetSocketAddress clientAddress;

    public VertxMqttConnection(MqttEndpoint endpoint) {
        this.endpoint = endpoint;
        this.keepAliveTimeoutMs = (long)(endpoint.keepAliveTimeSeconds() + 10) * 1000L;
    }

    @Override
    public Duration getKeepAliveTimeout() {
        return Duration.ofMillis(this.keepAliveTimeoutMs);
    }

    @Override
    public void onClose(Consumer<MqttConnection> listener) {
        this.disconnectConsumer = this.disconnectConsumer.andThen(listener);
    }

    @Override
    public Optional<MqttAuth> getAuth() {
        return this.endpoint.auth() == null ? Optional.of(emptyAuth) : Optional.of(new VertxMqttAuth());
    }

    @Override
    public void reject(MqttConnectReturnCode code) {
        if (this.closed) {
            return;
        }
        try {
            this.endpoint.reject(code);
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        try {
            this.complete();
        }
        catch (Throwable throwable) {
            // empty catch block
        }
    }

    @Override
    public Optional<MqttMessage> getWillMessage() {
        return Optional.ofNullable(this.endpoint.will()).filter(will -> will.getWillMessageBytes() != null).map(will -> SimpleMqttMessage.builder().will(true).payload(Unpooled.wrappedBuffer((byte[])will.getWillMessageBytes())).topic(will.getWillTopic()).qosLevel(will.getWillQos()).build());
    }

    @Override
    public MqttConnection accept() {
        if (this.accepted) {
            return this;
        }
        log.debug("mqtt client [{}] connected", (Object)this.getClientId());
        this.accepted = true;
        try {
            if (!this.endpoint.isConnected()) {
                this.endpoint.accept();
            }
        }
        catch (Exception e) {
            this.close().subscribe();
            log.warn(e.getMessage(), (Throwable)e);
            return this;
        }
        this.init();
        return this;
    }

    @Override
    public void keepAlive() {
        this.ping();
    }

    void ping() {
        this.lastPingTime = System.currentTimeMillis();
    }

    void init() {
        this.endpoint.disconnectHandler(ignore -> this.complete()).closeHandler(ignore -> this.complete()).exceptionHandler(error -> {
            if (error instanceof DecoderException && error.getMessage().contains("too large message")) {
                log.error("MQTT\u6d88\u606f\u8fc7\u5927,\u8bf7\u5728\u7f51\u7edc\u7ec4\u4ef6\u4e2d\u8bbe\u7f6e[\u6700\u5927\u6d88\u606f\u957f\u5ea6].", error);
                return;
            }
            log.error(error.getMessage(), error);
        }).pingHandler(ignore -> {
            this.ping();
            if (!this.endpoint.isAutoKeepAlive()) {
                this.endpoint.pong();
            }
        }).publishHandler(msg -> {
            boolean hasDownstream;
            this.ping();
            VertxMqttPublishing publishing = new VertxMqttPublishing((MqttPublishMessage)msg, false);
            boolean bl = hasDownstream = this.messageProcessor.currentSubscriberCount() > 0;
            if (this.autoAckMsg && hasDownstream) {
                publishing.acknowledge();
            }
            if (hasDownstream) {
                this.messageProcessor.emitNext((Object)publishing, Reactors.emitFailureHandler());
            }
        }).publishAcknowledgeHandler(messageId -> {
            this.ping();
            log.debug("PUBACK mqtt[{}] message[{}]", (Object)this.getClientId(), messageId);
        }).publishReceivedHandler(messageId -> {
            this.ping();
            log.debug("PUBREC mqtt[{}] message[{}]", (Object)this.getClientId(), messageId);
            this.endpoint.publishRelease(messageId.intValue());
        }).publishReleaseHandler(messageId -> {
            this.ping();
            log.debug("PUBREL mqtt[{}] message[{}]", (Object)this.getClientId(), messageId);
            this.endpoint.publishComplete(messageId.intValue());
        }).publishCompletionHandler(messageId -> {
            this.ping();
            log.debug("PUBCOMP mqtt[{}] message[{}]", (Object)this.getClientId(), messageId);
        }).subscribeHandler(msg -> {
            boolean hasDownstream;
            this.ping();
            VertxMqttSubscription subscription = new VertxMqttSubscription((MqttSubscribeMessage)msg, false);
            boolean bl = hasDownstream = this.subscription.currentSubscriberCount() > 0;
            if (this.autoAckSub || !hasDownstream) {
                subscription.acknowledge();
            }
            if (hasDownstream) {
                this.subscription.emitNext((Object)subscription, Reactors.emitFailureHandler());
            }
        }).unsubscribeHandler(msg -> {
            boolean hasDownstream;
            this.ping();
            VertxMqttMqttUnSubscription unSubscription = new VertxMqttMqttUnSubscription((MqttUnsubscribeMessage)msg, false);
            boolean bl = hasDownstream = this.unsubscription.currentSubscriberCount() > 0;
            if (this.autoAckUnSub || !hasDownstream) {
                unSubscription.acknowledge();
            }
            if (hasDownstream) {
                this.unsubscription.emitNext((Object)unSubscription, Reactors.emitFailureHandler());
            }
        });
    }

    @Override
    public void setKeepAliveTimeout(Duration duration) {
        this.keepAliveTimeoutMs = duration.toMillis();
    }

    @Override
    public InetSocketAddress getClientAddress() {
        try {
            SocketAddress address;
            if (this.clientAddress == null && this.endpoint != null && (address = this.endpoint.remoteAddress()) != null) {
                this.clientAddress = new InetSocketAddress(address.host(), address.port());
            }
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        return this.clientAddress;
    }

    @Override
    public String getClientId() {
        return this.endpoint.clientIdentifier();
    }

    @Override
    public Flux<MqttPublishing> handleMessage() {
        return this.messageProcessor.asFlux();
    }

    @Override
    public Mono<Void> publish(MqttMessage message) {
        this.ping();
        int messageId = message.getMessageId() <= 0 ? this.nextMessageId() : message.getMessageId();
        return Mono.create(sink -> {
            ByteBuf buf = message.getPayload();
            Buffer buffer = Buffer.buffer((ByteBuf)buf);
            this.endpoint.publish(message.getTopic(), buffer, MqttQoS.valueOf((int)message.getQosLevel()), message.isDup(), message.isRetain(), messageId, message.getProperties(), result -> {
                if (result.succeeded()) {
                    sink.success();
                } else {
                    sink.error(result.cause());
                }
                ReferenceCountUtil.safeRelease((Object)buf);
            });
        });
    }

    @Override
    public Flux<MqttSubscription> handleSubscribe(boolean autoAck) {
        this.autoAckSub = autoAck;
        return this.subscription.asFlux();
    }

    @Override
    public Flux<MqttUnSubscription> handleUnSubscribe(boolean autoAck) {
        this.autoAckUnSub = autoAck;
        return this.unsubscription.asFlux();
    }

    public InetSocketAddress address() {
        return this.getClientAddress();
    }

    public Mono<Void> sendMessage(EncodedMessage message) {
        if (message instanceof MqttMessage) {
            return this.publish((MqttMessage)message);
        }
        return Mono.empty();
    }

    public Flux<EncodedMessage> receiveMessage() {
        return this.handleMessage().cast(EncodedMessage.class);
    }

    public void disconnect() {
        this.close().subscribe();
    }

    @Override
    public boolean isAlive() {
        return this.endpoint.isConnected() && (this.keepAliveTimeoutMs < 0L || System.currentTimeMillis() - this.lastPingTime < this.keepAliveTimeoutMs);
    }

    @Override
    public Mono<Void> close() {
        if (this.closed) {
            return Mono.empty();
        }
        return Mono.fromRunnable(() -> {
            try {
                if (this.endpoint.isConnected()) {
                    this.endpoint.close();
                } else {
                    this.complete();
                }
            }
            catch (Throwable throwable) {
                // empty catch block
            }
        });
    }

    private void complete() {
        if (this.closed) {
            return;
        }
        this.closed = true;
        this.disconnectConsumer.accept(this);
    }

    private int nextMessageId() {
        this.messageIdCounter = this.messageIdCounter % 65535 != 0 ? this.messageIdCounter + 1 : 1;
        return this.messageIdCounter;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        VertxMqttConnection that = (VertxMqttConnection)o;
        return Objects.equals(this.endpoint, that.endpoint);
    }

    public int hashCode() {
        return Objects.hash(this.endpoint);
    }

    @Override
    public long getLastPingTime() {
        return this.lastPingTime;
    }

    class VertxMqttAuth
    implements MqttAuth {
        VertxMqttAuth() {
        }

        public String getUsername() {
            return VertxMqttConnection.this.endpoint.auth().getUsername();
        }

        public String getPassword() {
            return VertxMqttConnection.this.endpoint.auth().getPassword();
        }
    }

    class VertxMqttMqttUnSubscription
    implements MqttUnSubscription {
        private final MqttUnsubscribeMessage message;
        private volatile boolean acknowledged;

        @Override
        public MqttUnsubscribeMessage getMessage() {
            return this.message;
        }

        @Override
        public synchronized void acknowledge() {
            if (this.acknowledged) {
                return;
            }
            log.info("acknowledge mqtt [{}] unsubscribe : {} ", (Object)VertxMqttConnection.this.getClientId(), (Object)this.message.topics());
            this.acknowledged = true;
            VertxMqttConnection.this.endpoint.unsubscribeAcknowledge(this.message.messageId());
        }

        public VertxMqttMqttUnSubscription(MqttUnsubscribeMessage message, boolean acknowledged) {
            this.message = message;
            this.acknowledged = acknowledged;
        }
    }

    class VertxMqttSubscription
    implements MqttSubscription {
        private final MqttSubscribeMessage message;
        private volatile boolean acknowledged;

        @Override
        public MqttSubscribeMessage getMessage() {
            return this.message;
        }

        @Override
        public synchronized void acknowledge() {
            if (this.acknowledged) {
                return;
            }
            this.acknowledged = true;
            VertxMqttConnection.this.endpoint.subscribeAcknowledge(this.message.messageId(), this.message.topicSubscriptions().stream().map(MqttTopicSubscription::qualityOfService).collect(Collectors.toList()));
        }

        public VertxMqttSubscription(MqttSubscribeMessage message, boolean acknowledged) {
            this.message = message;
            this.acknowledged = acknowledged;
        }
    }

    class VertxMqttPublishing
    implements MqttPublishing {
        private final MqttPublishMessage message;
        private volatile boolean acknowledged;

        @Nonnull
        public String getTopic() {
            return this.message.topicName();
        }

        public String getClientId() {
            return VertxMqttConnection.this.getClientId();
        }

        public int getMessageId() {
            return this.message.messageId();
        }

        public boolean isWill() {
            return false;
        }

        public int getQosLevel() {
            return this.message.qosLevel().value();
        }

        public boolean isDup() {
            return this.message.isDup();
        }

        public boolean isRetain() {
            return this.message.isRetain();
        }

        @Nonnull
        public ByteBuf getPayload() {
            return this.message.payload().getByteBuf();
        }

        public String toString() {
            return this.print();
        }

        public MqttProperties getProperties() {
            return this.message.properties();
        }

        @Override
        public MqttMessage getMessage() {
            return this;
        }

        @Override
        public void acknowledge() {
            if (this.acknowledged) {
                return;
            }
            this.acknowledged = true;
            if (this.message.qosLevel() == MqttQoS.AT_LEAST_ONCE) {
                log.debug("PUBACK QoS1 mqtt[{}] message[{}]", (Object)this.getClientId(), (Object)this.message.messageId());
                VertxMqttConnection.this.endpoint.publishAcknowledge(this.message.messageId());
            } else if (this.message.qosLevel() == MqttQoS.EXACTLY_ONCE) {
                log.debug("PUBREC QoS2 mqtt[{}] message[{}]", (Object)this.getClientId(), (Object)this.message.messageId());
                VertxMqttConnection.this.endpoint.publishReceived(this.message.messageId());
            }
        }

        public VertxMqttPublishing(MqttPublishMessage message, boolean acknowledged) {
            this.message = message;
            this.acknowledged = acknowledged;
        }
    }
}

